import re
import os.path

from routersploit.core.exploit.exceptions import OptionValidationError
from routersploit.core.exploit.utils import (
    is_ipv4,
    is_ipv6,
)


class Option(object):
    """ Exploit attribute that is set by the end user """

    def __init__(self, default, description=""):
        self.label = None
        self.description = description

        self.default = default
        self.display_value = default
        self.value = default

    def __get__(self, instance, owner):
        return self.value
            
    def __set__(self, instance, value):
        if self._apply_widget(value):
            self.display_value = value
            self.value = value

    def set_label(self, label):
        self.label = label


class OptIP(Option):
    """ Option IP attribute """

    def _apply_widget(self, value):
        if is_ipv4(value) or is_ipv6(value):
            return value
        else:
            raise OptionValidationError("Invalid address. Provided address is not valid IPv4 or IPv6 address.")


class OptPort(Option):
    """ Option Port attribute """

    def _apply_widget(self, value):
        try:
            value = int(value)

            if 0 < value <= 65535:  # max port number is 65535
                return value
            else:
                raise OptionValidationError("Invalid option. Port value should be between 0 and 65536.")
        except ValueError:
            raise OptionValidationError("Invalid option. Cannot cast '{}' to integer.".format(value))
    

class OptBool(Option):
    """ Option Bool attribute """

    def _apply_widget(self, value):
        if value in ["true", "false"]:
            return value
        else:
            raise OptionValidationError("Invalid value. It should be true or false.")

    def __get__(self, instance, owner):
        if self.display_value == "true":
            return True

        return False


class OptInteger(Option):
    """ Option Integer attribute """

    def _apply_widget(self, value):
        try:
            return int(value)
        except ValueError:
            raise OptionValidationError("Invalid option. Cannot cast '{}' to integer.".format(value))


class OptFloat(Option):
    """ Option Float attribute """

    def _apply_widget(self, value):
        try:
            return float(value)
        except ValueError:
            raise OptionValidationError("Invalid option. Cannot cast '{}' to float.".format(value))


class OptString(Option):
    """ Option String attribute """

    def _apply_widget(self, value):
        try:
            return str(value)
        except ValueError:
            raise OptionValidationError("Invalid option. Cannot cast '{}' to string.".format(value))


class OptMAC(Option):
    """ Option MAC attribute """

    def _apply_widget(self, value):
        regexp = r"^[a-f\d]{1,2}:[a-f\d]{1,2}:[a-f\d]{1,2}:[a-f\d]{1,2}:[a-f\d]{1,2}:[a-f\d]{1,2}$"
        if re.match(regexp, value.lower()):
            return value
        else:
            raise OptionValidationError("Invalid option. '{}' is not a valid MAC address".format(value))


class OptWordlist(Option):
    """ Option Wordlist attribtue """

    def _apply_widget(self, value):
        if value.startswith("file://"):
            path = value.replace("file://", "")
            if not os.path.exists(path):
                raise OptionValidationError("File '{}' does not exist.".format(path))
            return value

        return value

    def __get__(self, instance, owner):
        if self.display_value.startswith("file://"):
            path = self.display_value.replace("file://", "")
            with open(path, "r") as f:
                lines = [line.strip() for line in f.readlines()]
                return lines

        return self.display_value.split(",")

    def __set__(self, instance, value):
        self.display_value = value
